// The MT63ASC receiver in C++ for LINUX, written to be compatible
// with the MT63ASC.ASM modem for the EVM56K/DSPCARD4.

// (c) 1999 Pawel Jalocha, SP9VRC
// Date: 08-NOV-1999

#include <stdio.h>

#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>

#ifdef __MSDOS__
#include <conio.h>
#endif

#include "dsp.h"
#include "mt63.h"
#include "sound.h"

#include "stdinr.h"

// ============================================================================

char DevName[32]=""; 	// the name of the sound device
char UseDevice=0;

int ReqRate=8000;	// we request 8000 Hz sampling
float DevRate=0;	// the device gives us a possibly different rate
float UserRate=0.0;	// user provided rate (overrides the DevRate)

int ReadFromFile=0;	// read real audio or playback from a file
char *PlaybackFile=NULL; // file to read audio from in playback mode

int SaveToFile=0;	// save audio to a file
char *SaveFile=NULL;	// file to save audio

/*
int SendTextFile=0;	// send text from a file
char *TextFile=NULL;
*/

int Bandwidth=1000;	// 500, 1000 or 2000 Hz bandwidth
int Interleave=0;	// short/long interleave
int IntegLen=16;        // integration period for sync./data tracking

// ============================================================================

SoundDevice RxDev;	// sound device to read audio

RateConvBL RateConv;	// rate converter to adjust the sampling rate

#define BuffLen 512
s16 InpBuff[BuffLen];	// Signed 16-bit audio input buffer
float_buff InpFloat;

MT63rx Rx;

// ============================================================================

char StatusLine[80];

main(int argc, char *argv[])
{ int err,arg,InpLen,i; char code;
  float ValueF; int ValueI;
  int OutFileNo=0; int wr;

  int Conf,X,Y; float SNR;

printf("\n\
Multitone MT63 modem for Linux - receiver, (c) 1999 Pawel Jalocha\n\
Made to be compatible with the MT63ASC for the EVM56K/DSPCARD4\n\
\n");

if(argc<=1)
{ printf("\
Usage: mt63rx <options>\n\
 the options:\n\
	 -d<device> the output dsp device number or name [default = none]\n\
		    -d => /dev/dsp, -d1 => /dev/dsp1, etc.\n\
		    or: -d/dev/dsp, -d/dev/dsp1, etc.\n\
	 -s<file>   save audio to a file\n\
	 -p<file>   playback audio from a file\n\
	 -r<rate>   request given sampling rate from device [8000 Hz]\n\
	 -R<rate>   the true sampling rate (if you know it already)\n\
	 -i         use the double interleave [default = single]\n\
	 -I<period> set the sync. integration period [16 symbols]\n\
	 -B<band>   select bandwidth: 500, 1000 or 2000 Hz [1000Hz]\n\
Examples:\n\
mt63rx -d0    => read audio from /dev/dsp0, decode with single interleave\n\
mt63rx -d0 -i -I32 => decode with double interleave and longer integration\n\
mt63rx -pmsg.raw => decode the audio file in the raw signed 16-bit format\n\
mt63rx -pmsg.raw -R8010 => decode from msg.raw, the sampling rate was 8010 Hz\n\
mt63rx -d0 -slog.raw -i => decode audio from /dev/dsp0 with double interleave\n\
			   but save it to log.raw for later playback\n\
");
goto Error; }

for(err=0,arg=1; arg<argc; arg++)
{ if(argv[arg][0]!='-')
  { printf("Unknown option %s\n",argv[arg]); err+=1; continue; }
  switch(argv[arg][1])
  {
  case 'd':
    if(argv[arg][2]=='\0')
    { strcpy(DevName,"/dev/dsp"); UseDevice=1; }
    else if(isdigit(argv[arg][2]))
    { strcpy(DevName,"/dev/dsp"); strcat(DevName,argv[arg]+2); UseDevice=1; }
    else if(argv[arg][2]=='/')
    { strcpy(DevName,argv[arg]+2); UseDevice=1; }
    else
    { printf("Bad device number/name in %s\n",argv[arg]); err+=1; }
    break;
  case 'r':
    if(sscanf(argv[arg]+2,"%d",&ValueI)!=1)
      { printf("Invalid decimal number in %s\n",argv[arg]); err+=1; }
    else ReqRate=ValueI;
    break;
  case 'I':
    if(sscanf(argv[arg]+2,"%d",&ValueI)!=1)
      { printf("Invalid decimal number in %s\n",argv[arg]); err+=1; }
    else IntegLen=ValueI;
    break;
  case 'R':
    if(sscanf(argv[arg]+2,"%f",&ValueF)!=1)
      { printf("Invalid floating point number in %s\n",argv[arg]); err+=1; }
    else UserRate=ValueF;
    break;
  case 's':
    SaveToFile=1; SaveFile=argv[arg]+2; break;
  case 'p':
    ReadFromFile=1; PlaybackFile=argv[arg]+2; break;
  case 'i':
    Interleave=1;
    break;
  case 'B':
    if(sscanf(argv[arg]+2,"%d",&ValueI)!=1)
      { printf("Invalid decimal number in %s\n",argv[arg]); err+=1; }
    else
    { switch(ValueI)
      { case 500:
	case 1000:
	case 2000:
	  Bandwidth=ValueI; break;
	default:
	  { printf("The modem bandwidth can only be 500, 1000 or 2000 Hz\n"); err+=1; }
      }
    } break;
  default:
    printf("Unknown option %s\n",argv[arg]); err+=1;
  }
}

if(err) goto Error;

if((UseDevice==0)&&(ReadFromFile==0))
{ printf("No audio device selected nor a file to read audio\n"); goto Error; }

if(UseDevice) printf("Audio device for input is %s\n",DevName);
else printf("No audio device selected\n");

if(ReadFromFile)
{ if(UseDevice)
  { err=RxDev.OpenFileForRead(PlaybackFile,ReqRate,DevName);
    if(err<0)
    { printf("Can't open %s or %s to read audio from\n",PlaybackFile,DevName); goto Error; }
    printf("We read the audio from %s and play it on %s\n",
	    PlaybackFile,DevName);
  } else
  { err=RxDev.OpenFileForRead(PlaybackFile,ReqRate);
    if(err<0)
    { printf("Can't open %s to read audio from\n",PlaybackFile); goto Error; }
    printf("We read the audio from %s\n",PlaybackFile);
  }
} else
{ err=RxDev.OpenForRead(DevName,ReqRate);
  if(err<0) { printf("Can't open %s to read audio from: errno => %s\n",
		   DevName,strerror(errno)); goto Error; }
}

if(SaveToFile)
{
#ifdef __MSDOS__
  OutFileNo=open(SaveFile,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY,S_IREAD|S_IWRITE);
#else
  OutFileNo=open(SaveFile,O_WRONLY|O_CREAT|O_TRUNC,S_IREAD|S_IWRITE);
#endif
  if(OutFileNo<0)
  { printf("Can't open file %s to save audio, errno => %s\n",
	   SaveFile,strerror(errno)); goto Error; }
  printf("We will save the audio to %s for later playback\n",SaveFile);
} else OutFileNo=0;

DevRate=RxDev.Rate;
printf("Requested rate = %d Hz, device reported rate = %1.0f Hz\n",
       ReqRate,DevRate);
if(UserRate>0.0)
{ printf("You say the true device rate is %4.2f Hz.\n",UserRate);
  DevRate=UserRate; }

err=RateConv.Preset(32,NULL,16);
if(err)
{ printf("Can't preset the rate converter - not enough RAM ?!\n"); goto Error; }
err=RateConv.ComputeShape(0.0,M_PI,WindowHamming);
if(err)
{ printf("Can't compute the shape for the rate converter - not enough RAM ?!\n"); goto Error; }
RateConv.SetOutVsInp(8000.0/DevRate);

printf("Modem bandwidth is %d Hz with %s interleave\n",
	Bandwidth,Interleave ? "DOUBLE (64)" : "SINGLE (32)");
printf("The time/frequency synchronizer integrates over %d symbols\n",IntegLen);

err=Rx.Preset(Bandwidth,Interleave,IntegLen);
if(err) { printf("Can't preset the MT63 receiver - not enough RAM ?\n");
	  goto Error; }

/*
InpFloat.EnsureSpace(BuffLen);
for(i=0; i<BuffLen; i++) InpFloat.Data[i]=0.0;
InpFloat.Len=200;		// timing shift for debug
RateConv.ProcessLinI(&InpFloat);
Rx.Process(&RateConv.Output);
*/
printf("\nPress RETURN to stop\n\n");
while(StdinReady()==0)
{ InpLen=RxDev.Read(InpBuff,BuffLen);
  if(InpLen<0)
  { printf("Error while reading audio from %s, errno => %s\n",
	   DevName,strerror(errno)); break; }
// the following applies when we read audio in non-blocking mode
#ifdef __linux__
  if(InpLen==0) { usleep(50000); continue; }
#endif
#ifdef __MSDOS__
  if(InpLen==0) continue;
#endif

  if(OutFileNo>0)
  { wr=write(OutFileNo,InpBuff,2*InpLen);
    if(wr!=(2*InpLen))
    { printf("Error while writing data to %s\n",SaveFile); break; }
  }

  ConvS16toFloat(InpBuff,&InpFloat,InpLen);
  RateConv.ProcessLinI(&InpFloat);
  Rx.Process(&RateConv.Output);
  for(i=0; i<Rx.Output.Len; i++)
  { code=Rx.Output.Data[i];
    if((code>=' ')||(code=='\n')||(code=='\r')) printf("%c",code);
    else if(code!='\0') printf("<%02X>",code);
    fflush(stdout);
  }
  Conf=(int)floor(Rx.SYNC_Confidence()*100.0); if(Conf>99) Conf=99;
  SNR=Rx.FEC_SNR(); if(SNR>99.9) SNR=99.9;
  sprintf(StatusLine,"Inp:%5.3f/%04.1f%% %s:%02d%%/%+05.2f/%4.2f FEC:%4.1f/%+2d %+6.1fHz ",
    RMS(&InpFloat), 100.0*(InpFloat.Len-CountInRange(&InpFloat,-0.5,0.5))/InpFloat.Len,
    Rx.SYNC_LockStatus() ? "Lock" : "Seek",Conf,Rx.SYNC_FreqOffset(),Rx.SYNC_TimeOffset(),
    SNR,Rx.FEC_CarrOffset(), Rx.TotalFreqOffset());
#ifdef __MSDOS__
  X=wherex(); Y=wherey();
  gotoxy(1,1); textcolor(BLACK); textbackground(LIGHTGRAY); cprintf("%s",StatusLine);
  gotoxy(X,Y); textcolor(LIGHTGRAY); textbackground(LIGHTGRAY);
#endif

}

Stop:
 printf("\nClosing audio device ...\n");
 RxDev.Close();
 if(OutFileNo>0) close(OutFileNo);
 printf("Stopped OK.\n");
 printf("%ld samples read = %3.1f sec\n",
     RxDev.TotalRead,RxDev.TotalRead/DevRate);
 return 0;

Error:
 RxDev.Close();
 if(OutFileNo>0) close(OutFileNo);
 return 1;

}
